/* 
 * Copyright 2015-2022, 2026 The Regents of the University of California
 * All rights reserved.
 * 
 * This file is part of Spoofer.
 * 
 * Spoofer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Spoofer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Spoofer.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <string>
#include <sstream>
#include <exception>
#include <stdexcept>
#include <cstdio>
#include <time.h>
#include "spoof_qt.h"
#include <QDebug>
#include <QtDebug>
#include <QCommandLineParser>
#include <QLocalServer>
#include <QLocalSocket>
#include <QProcess>
#include <QStandardPaths>
#include <QDir>
#ifdef ENABLE_QNetworkConfigurationManager
#include <QNetworkConfigurationManager> // experimental
#endif
#include <QNetworkInterface>
#include <QCoreApplication>
#include <QLockFile>
#include <QRegularExpression>
#include <QTextStream>
#include <QFileSystemWatcher>
#include <QThread>
#include "../../config.h"
#include "app.h"
#include "BlockWriter.h"


const QString App::appname(QSL(APPNAME));
const QString App::schedulerLogFtime = QSL("'" APPNAME "-'yyyy~MM~dd-HH~mm~ss'.txt'");
const QString App::schedulerLogGlob = QSL(APPNAME "-\?\?\?\?\?\?\?\?-\?\?\?\?\?\?.txt"); 
QString App::defaultDataDir;
QString App::dataDir;

#if defined(Q_OS_WIN32)
#   include "appwin.h"
    App *App::newapp(int &argc, char **argv) { return new AppWin(argc, argv); }
#elif defined(Q_OS_MACOS) // must come before Q_OS_UNIX
#   include "appmac.h"
    App *App::newapp(int &argc, char **argv) { return new AppMac(argc, argv); }
#elif defined(Q_OS_UNIX)
#   include "appunix.h"
    App *App::newapp(int &argc, char **argv) { return new AppUnix(argc, argv); }
#endif

static void copySettings(QSettings &dst, const QSettings &src)
{
    const QStringList keys = src.allKeys();
    for (const QString &key : keys) {
	dst.setValue(key, src.value(key));
    }
}

QString App::AppLog::makeName()
{
    QDir dir(dataDir);
    return dir.filePath(ftime_utc(schedulerLogFtime));
}

QList<SubnetAddr> App::getAddresses() const
{
    QList<SubnetAddr> addrs;
    const QList<QNetworkInterface> ifaces = QNetworkInterface::allInterfaces();
    for (const QNetworkInterface &iface : ifaces) {
	if (!iface.isValid()) continue;
	if (!(iface.flags() & QNetworkInterface::IsRunning)) continue;
	if (!(iface.flags() & QNetworkInterface::IsUp)) continue;
	if ((iface.flags() & QNetworkInterface::IsLoopBack)) continue;
	const QList<QNetworkAddressEntry> addrentries = iface.addressEntries();
	for (const QNetworkAddressEntry &addrentry : addrentries) {
	    QHostAddress addr = addrentry.ip();
	    if (addr.isNull()) continue;
	    if (addr.isLoopback()) continue;
	    if (!addr.scopeId().isEmpty()) continue; // link-local or site-local
	    if (addr.protocol() == QAbstractSocket::IPv4Protocol) {
		if (!config->enableIPv4()) continue;
		if (addrentry.prefixLength() == 32)
		    continue; // skip address with full-length mask
	    } else if (addr.protocol() == QAbstractSocket::IPv6Protocol) {
		if (!config->enableIPv6()) continue;
		if (addrentry.prefixLength() == 128)
		    continue; // skip address with full-length mask
	    }
	    addrs.push_back(SubnetAddr(addr, addrentry.prefixLength()));
	}
    }
    return addrs;
}

void App::handleNetChange()
{
    qDebug() << "possible network change detected" << qPrintable(ftime_utc());
    proberTimer.stop(); // Temporarily prevent prober from starting.
    // Do NOT reset nextProberStart.when here.  If scheduleNextProber() finds
    // no relevant change in the network config, it will restart the timer to
    // resume the original schedule.
    if (prober) return; // prober is running now
    // Wait 1s (in case of a burst of changes), then scheduleNextProber()
    netPollTimer.start(1000*1);
}

void App::scheduleNextProber()
{
    time_t now, nextProbe = 0, nextPoll = 0;
    if (prober) return; // prober is running now
    if (paused || !config->hasRequiredSettings() || upgradeState > US_NONE) return;
    proberTimer.stop();
    netPollTimer.stop();
    time(&now);
    nextPoll = now + config->netPollInterval();

    const QList<SubnetAddr> addrs = getAddresses();
    QSet<SubnetAddr> subnets;
    for (const SubnetAddr &addr : addrs) {
	subnets.insert(addr.prefix());
    }

    if (nextProberStart.when && subnets == scheduledSubnets) {
	// No change.  Resume original schedule.
	qDebug() << "scheduleNextProber: no relevant change in network config";
	nextProbe = nextProberStart.when;

    } else if (!subnets.isEmpty()) {
	for (const SubnetAddr &subnet : subnets) {
	    if (!scheduledSubnets.contains(subnets))
		sperr << "new subnet detected: " << subnet.toString() << Qt_endl;
	}
	qDebug() << "scheduleNextProber" << qPrintable(ftime_utc(QString(), &now));
	for (const SubnetAddr &subnet : subnets) {
	    const RunRecord &rr = pastRuns[subnet];
	    time_t interval = config->proberInterval();
	    if (rr.errors > 0 && rr.errors <= config->maxRetries()) {
		// exponential backoff, capped at proberInterval
		interval = config->proberRetryInterval();
		for (int i = 1; i < rr.errors; i++) {
		    interval <<= 1;
		    if (interval >= config->proberInterval()) {
			interval = config->proberInterval();
			break;
		    }
		}
	    }
	    time_t t = rr.t + interval;
	    if (!nextProbe || t < nextProbe) nextProbe = t;
	}
	qDebug() << "  earliest" << qPrintable(ftime_utc(QString(), &nextProbe));
	if (nextProbe < now + config->delayInterval())
	    nextProbe = now + config->delayInterval();
	// Re-check the network configuration <delayInterval> before the next
	// scheduled run, or after <netPollInterval>, whichever is earlier.
	if (nextPoll > nextProbe - config->delayInterval() && nextProbe - config->delayInterval() > now)
	    nextPoll = nextProbe - config->delayInterval();
    }

    scheduledSubnets = subnets;

    if (!nextProbe || nextProbe > nextPoll) {
	// Schedule another netPoll before the next prober.
	netPollTimer.start(1000 * static_cast<int>(nextPoll - now));
    } else {
	// Schedule a prober before the next netPoll.
	// NB: more than ~24 days would overflow proberTimer.start()'s int
	// parameter (but nextPoll is guaranteed to earlier than that).
	proberTimer.start(nextProbe <= now ? 1 : 1000 * static_cast<int>(nextProbe - now));
    }

    QString label = nextProbe ? ftime_utc(QString(), &nextProbe) : QSL("never");
    if (nextProbe != nextProberStart.when) {
	sperr << "next prober run: " << label << Qt_endl;
	nextProberStart.when = safe_int<qint32>(nextProbe);
	for (QLocalSocket *ui : uiSet) {
	    if (opAllowed(ui, config->unprivView))
		BlockWriter(ui) << (qint32)SC_SCHEDULED << nextProberStart;
	}
    } else {
	qDebug() << "next prober run:" << qPrintable(label) << "(silent)";
    }
}

void App::dumpNetCfg(QLocalSocket *ui)
{
    const QList<SubnetAddr> addrs = getAddresses();
    for (const SubnetAddr &addr : addrs) {
	QString msg =
	    (addr.addr().protocol() == QAbstractSocket::IPv4Protocol ? QSL("ipv4") :
	     addr.addr().protocol() == QAbstractSocket::IPv6Protocol ? QSL("ipv6") :
	     QSL("????")) % QSL(": ") %
	    addr.toString() % QSL("  ") % addr.prefix().toString();
	BlockWriter(ui) << (qint32)SC_TEXT << msg;
    }
}

bool App::parseCommandLine(QCommandLineParser &clp)
{
    QCommandLineOption cloDetached(QStringList() << QSL("D") <<
#ifdef Q_OS_WIN32
	QSL("detach"), QSL("Detach from console, if there is one.")
#else
	QSL("daemon"), QSL("Run as a daemon.")
#endif
	);
    clp.addOption(cloDetached);

    QCommandLineOption cloSharePublic(
	QStringList() << QSL("s") << QSL("share-public"),
	config->sharePublic.optionHelpString(), QSL("1|0"));
    clp.addOption(cloSharePublic);

    QCommandLineOption cloShareRemedy(
	QStringList() << QSL("r") << QSL("share-remedy"),
	config->shareRemedy.optionHelpString(), QSL("1|0"));
    clp.addOption(cloShareRemedy);

    QCommandLineOption cloStartPaused(
	QStringList() << QSL("p") << QSL("paused"),
	config->paused.optionHelpString(), QSL("1|0"));
    clp.addOption(cloStartPaused);

    QCommandLineOption cloInitOnly(
	QSL("init"),
	QSL("Store the values of the --datadir, --paused, --share-public, and/or --share-remedy options to persistent settings, and exit."));
    clp.addOption(cloInitOnly);

    QCommandLineOption cloDeleteUpgrade(
	QSL("delete-upgrade"),
	QSL("Delete pending upgrade information, and exit."));
    clp.addOption(cloDeleteUpgrade);

    QCommandLineOption cloDeleteData(
	QSL("delete-data"),
	QSL("Delete all data and exit."));
    clp.addOption(cloDeleteData);

    QCommandLineOption cloDumpSettings(
	QSL("dump-settings"),
	QSL("Print current settings and exit."));
    clp.addOption(cloDumpSettings);

    QCommandLineOption cloCheckSettings(
	QSL("check-settings"),
	QSL("Exit with status 0 if all required settings are set, nonzero otherwise."));
    clp.addOption(cloCheckSettings);

    QCommandLineOption cloDeleteSettings(
	QSL("delete-settings"),
	QSL("Delete all settings and exit."));
    clp.addOption(cloDeleteSettings);

    QCommandLineOption cloSaveSettings(
	QSL("save-settings"),
	QSL("Save a copy of settings to <file> in INI format, and exit."),
	QSL("file"));
    clp.addOption(cloSaveSettings);

    QCommandLineOption cloRestoreSettings(
	QSL("restore-settings"),
	QSL("Restore settings from <file> (created with --save-settings), and exit."),
	QSL("file"));
    clp.addOption(cloRestoreSettings);

    QCommandLineOption cloDumpPaths(
	QSL("dump-paths"),
	QSL("Print list of QStandardPaths, for debugging."));
    clp.addOption(cloDumpPaths);

    QCommandLineOption cloLogfile(
	QStringList() << QSL("l") << QSL("logfile"),
	QSL("Log to \"<datadir>/%1-<time>.txt\".").arg(App::appname));
    clp.addOption(cloLogfile);

    QCommandLineOption cloDataDir(
	QStringList() << QSL("d") << QSL("datadir"),
	config->dataDir.optionHelpString(), QSL("dir"));
    clp.addOption(cloDataDir);

    if (!SpooferBase::parseCommandLine(clp, QSL("Spoofer scheduler service")))
	return false;

    this->optDeleteUpgrade = clp.isSet(cloDeleteUpgrade);
    this->optDeleteData = clp.isSet(cloDeleteData);
    this->optDeleteSettings = clp.isSet(cloDeleteSettings);
    this->optDumpPaths = clp.isSet(cloDumpPaths);
    this->optDumpSettings = clp.isSet(cloDumpSettings);
    this->optCheckSettings = clp.isSet(cloCheckSettings);
    this->optDetach = clp.isSet(cloDetached);
    this->optLogfile = clp.isSet(cloLogfile);
    this->optInitOnly = clp.isSet(cloInitOnly);

    if (clp.isSet(cloDataDir))
	this->optDataDir = QDir::current().absoluteFilePath(clp.value(cloDataDir));

    if (clp.isSet(cloSaveSettings)) {
	this->optSaveSettings = true;
	this->altSettingsFile = QDir::current().absoluteFilePath(clp.value(cloSaveSettings));
    }

    if (clp.isSet(cloRestoreSettings)) {
	this->optRestoreSettings = true;
	this->altSettingsFile = QDir::current().absoluteFilePath(clp.value(cloRestoreSettings));
    }

    if (!parseOptionFlag(optSharePublic, clp, cloSharePublic))
	return false;

    if (!parseOptionFlag(optShareRemedy, clp, cloShareRemedy))
	return false;

    if (!parseOptionFlag(optStartPaused, clp, cloStartPaused))
	return false;

    return true;
}

bool App::parseOptionFlag(int &opt, QCommandLineParser &clp,
    const QCommandLineOption &clo)
{
    if (!clp.isSet(clo))
	return true;
    QString value = clp.value(clo);
    if (value == QSL("0"))
	opt = 0;
    else if (value == QSL("1"))
	opt = 1;
    else {
	sperr << "illegal value for \"" << clo.names().last() << "\" option: " << value << Qt_endl;
	return false;
    }
    return true;
}

void App::dumpPaths(void) const
{
    spout << "applicationDirPath: " << "\"" << appDir << "\"" << Qt_endl;
    spout << "applicationFilePath: " << "\"" << appFile << "\"" << Qt_endl;
    spout << "homePath: " << "\"" << QDir::homePath() << "\"" << Qt_endl;

#define dumpQtPaths(id) \
    do { \
	QStandardPaths::StandardLocation type = QStandardPaths:: id ## Location; \
	spout << (int)type << " " << #id << " # " << QStandardPaths::displayName(type) << Qt_endl; \
	QStringList list = QStandardPaths::standardLocations(type); \
	for (int i = 0; i < list.size(); ++i) { \
	    spout << "    \"" << list.at(i) << "\"" << Qt_endl; \
	} \
    } while (0)

    dumpQtPaths(Temp          );          // 7
    dumpQtPaths(Home          );          // 8
    dumpQtPaths(Data          );          // 9
    dumpQtPaths(GenericData   );          // 11
    dumpQtPaths(Runtime       );          // 12
    dumpQtPaths(Config        );          // 13
    dumpQtPaths(GenericConfig );          // 16
}

void App::dumpSettings(void) const
{
    spout << "Settings in " << config->fileName() << Qt_endl;
    spout << "  " << qSetFieldWidth(25) << Qt_left << "" << "current value" <<
	"default value" << qSetFieldWidth(0) << Qt_endl;
    for (const Config::MemberBase *m : config->members) {
	spout << "  " << qSetFieldWidth(25) << m->key;
	if (m->isSet()) 
	    spout << m->variant().toString();
	else if (m->required)
	    spout << QSL("(unset, required)");
	else
	    spout << QSL("(unset)");
	spout << m->defaultVal.toString();
	spout << qSetFieldWidth(0) << Qt_endl;
    }
}

QString App::chooseDataDir()
{
    // QStandardPaths::standardLocations(QStandardPaths::DataLocation) may
    // return multiple types of locations, in no particular order:
    //   1 global:    linux example:  /usr/share/CAIDA/Spoofer        
    //   2 app base:                  (directory of binary)
    //   3 user:      linux example:  $HOME/.local/share/CAIDA/Spoofer
    // We prefer them in the order listed above.
    QString result;
    int best_score = -1;
    QString appBase = appDir;

#ifdef Q_OS_UNIX
    // Set HOME env var to a value that we can recognize when
    // standardLocations() inserts it into a path.
    const char *envHome = getenv("HOME");
    if (envHome) envHome = strdup(envHome);
    setenv("HOME", "/@HOME@", 1);
#endif

    const QStringList paths =
	QStandardPaths::standardLocations(QStandardPaths::DataLocation);
#ifdef Q_OS_UNIX
    // Restore HOME.
    if (envHome)
	setenv("HOME", envHome, 1);
    else
	unsetenv("HOME");
#endif
    qDebug() << "homePath:" << QDir::homePath();
    qDebug() << "rootPath:" << QDir::rootPath();
    for (const QString &path : paths) {
	if (path.isEmpty()) continue;
	QString cleanpath = QDir::cleanPath(path);
	int score = 1;
	if (cleanpath.startsWith(appBase))
	    score = 2;
	else if (cleanpath.startsWith(QSL("/@HOME@")))
	    score = 3;
	// In case standardLocations() doesn't use $HOME to generate paths, we
	// check for the real home path in addition to our "/@HOME@" token.
	else if (QDir::homePath() != QDir::rootPath() &&
	    cleanpath.startsWith(QDir::homePath()))
		score = 3;
	qDebug() << "dataDir option, score" << score << ":" << cleanpath;
	if (result.isEmpty() || (score < best_score)) {
	    best_score = score;
	    result = cleanpath;
	}
    }
    result.replace(QSL("/@HOME@"), QDir::homePath());
    return result;
}

QLocalServer *App::listen(bool privileged)
{
    QLocalServer *server = new QLocalServer(this);
    connect(server, &QLocalServer::newConnection, this, &App::uiAccept);
    
    server->setSocketOptions(
#ifndef EVERYONE_IS_PRIVILEGED
	privileged ? QLocalServer::UserAccessOption :
#endif
	QLocalServer::WorldAccessOption);

    QString serverName =
#ifdef Q_OS_UNIX
	// Use our own directory because the default directory may not be
	// readable by everyone (seen in Qt 5.9 on macOS 10.12)
	dataDir % QSL("/") %
#endif
	QSL("spoofer-") % QString::number(applicationPid());
    if (!privileged) serverName.append(QSL("o"));
    if (!server->listen(serverName)) {
	sperr << "listen on " << serverName << ": " << server->errorString() << Qt_endl;
	delete server;
	return nullptr;
    }
    sperr << "Scheduler listening for " <<
#ifndef EVERYONE_IS_PRIVILEGED
	(privileged ? "privileged " : "unprivileged ") <<
#endif
	"connections on " << server->fullServerName() << Qt_endl;
    return server;
}

QLockFile *App::lockLockFile(QString name)
{
    qDebug() << "lockFile:" << qPrintable(QDir::toNativeSeparators(name));
    QDir().mkpath(QDir(name % QSL("/..")).absolutePath()); // make sure dir exists
    QLockFile *lockfile = new QLockFile(name);
    lockfile->setStaleLockTime(0);
    int tries = 0;
    qint64 myPid;
    char buf[1024];
retry:
    if (++tries > 3)
	return nullptr;
    if (tries > 1)
	QThread::msleep(1000);
    if (lockfile->tryLock(0)) {
	if (tries > 1)
	    sperr << "Locked \"" <<  QDir::toNativeSeparators(name) <<
		"\" after " << tries << " tries" << Qt_endl;
	return lockfile;
    }
    error_t syserr = getLastErr();
    qint64 lockPid = 0;
    QString lockHost, lockApp;
    sperr << "Error locking \"" << QDir::toNativeSeparators(name) << "\":  ";
    if (syserr != 0)
	sperr << getLastErrmsg() << ".  ";
    switch (lockfile->error()) {
    case QLockFile::LockFailedError:
#ifdef Q_OS_WIN32
	// on windows, we can get this instead of QLockFile::PermissionError
	if (syserr == ERROR_ACCESS_DENIED) { sperr << Qt_endl; break; }
#endif
	myPid = QCoreApplication::applicationPid();
	if (!lockfile->getLockInfo(&lockPid, &lockHost, &lockApp)) {
	    // Either the lock file was removed between tryLock() and now,
	    // or it is corrupt.
	    sperr << "Unable to get lock information." << Qt_endl;
	} else {
	    sperr << "Locked by pid " << lockPid << " " << lockApp << Qt_endl;
	    // tryLock() may fail to remove a stale lock file if a new process
	    // (maybe even this one) got the same pid as the dead locking
	    // process.  This is especially likely if the old and new
	    // processes were started in a boot script.
	    if (lockPid == myPid) {
		sperr << "... but I am pid " << myPid << "." << Qt_endl;
	    } else if (!getProcessName(lockPid, buf, sizeof(buf))) {
		sperr << "... but there is no pid " << lockPid << " (" <<
		    getLastErrmsg() << ")" << Qt_endl;
	    } else if (!*buf) {
		sperr << "... and name of pid " << lockPid << " is unknown (" <<
		    getLastErrmsg() << ")" << Qt_endl;
	    } else {
#ifdef Q_OS_WIN32
		bool stale = (strcasecmp(buf, APPNAME ".exe") != 0);
#else
		const char *p = strstr(buf, APPNAME);
		bool stale = !(p && (p == buf || p[-1] == '/'));
#endif
		sperr << (stale ? "... but pid " : "... and pid ") <<
		    lockPid << (stale ? " is actually " : " is ") <<
		    buf << Qt_endl;
		if (!stale) break;
	    }
	}

	if (lockfile->removeStaleLockFile()) {
	    sperr << "Stale lock file removed." << Qt_endl;
	    goto retry;
	}
	syserr = getLastErr();
#ifdef Q_OS_WIN32
	if (syserr == ERROR_FILE_NOT_FOUND)
#else
	if (syserr == ENOENT)
#endif
	{
	    sperr << "Old lock file is already gone." << Qt_endl;
	    goto retry;
	}
	sperr << "Failed to remove existing lock file: " <<
	    getErrmsg(syserr) << Qt_endl;
	break;
    case QLockFile::PermissionError:
	sperr << "Permission error." << Qt_endl;
	break;
    case QLockFile::NoError:
	sperr << "No error." << Qt_endl;
	break;
    case QLockFile::UnknownError:
    default:
	sperr << "Unknown error (" << (unsigned)lockfile->error() << ")" <<
            Qt_endl;
	break;
    }
    delete lockfile;
    return nullptr;
}

static void removeFiles(QDir dir, QString glob, int keep_n = 0)
{
    QStringList filenames = dir.entryList(QStringList() << glob,
	QDir::Files, QDir::Name);
    for (int i = 0; i < filenames.size() - keep_n; i++) {
	qDebug() << "removing file" << filenames.at(i);
	if (!dir.remove(filenames.at(i))) {
	    App::sperr << "error removing file " << filenames.at(i) << ": " <<
		getLastErrmsg() << Qt_endl;
	}
    }
}

// Returns true if caller should continue with app.exec().
bool App::init(int &exitCode)
{
    exitCode = SP_EXIT_FAIL; // default result is failure
    connect(this, &QCoreApplication::aboutToQuit, this, &App::cleanup);
    QCommandLineParser clp;
    defaultDataDir = chooseDataDir();
    config->dataDir.setDefault(defaultDataDir);

    // Before chdir, so relative paths on cmdline refer to initial directory
    if (!parseCommandLine(clp)) {
	exitCode = SP_EXIT_CMDLINE;
	return false;
    }
    if (!initConfig(!optDumpSettings)) {
	exitCode = SP_EXIT_SETTINGS;
	return false;
    }

    if (optDumpSettings) {
	dumpSettings();
	exitCode = SP_EXIT_OK; // success
	return false;
    }

    if (optCheckSettings) {
	exitCode = config->hasRequiredSettings() ? SP_EXIT_OK : SP_EXIT_FAIL;
	return false;
    }

    if (optDumpPaths) dumpPaths();

    if (!optDataDir.isEmpty()) {
	dataDir = optDataDir;
    } else if (!config->dataDir().isEmpty()) {
	dataDir = config->dataDir();
    } else if (!defaultDataDir.isEmpty()) {
	dataDir = defaultDataDir;
    } else {
	sperr << "can't determine a suitable data folder" << Qt_endl;
	exitCode = SP_EXIT_DATADIR;
	return false;
    }

    QDir dir;
    dir.mkpath(dataDir); // make sure it exists
    // don't "config->dataDir(dataDir)" until we have lock and are listening

    // Maybe open new log file using new value of dataDir
    if (optLogfile || errdev.type() == typeid(AppLog)) {
	errdev.setDevice(new AppLog());
    } else {
	errdev.setFallbackDevice(new AppLog());
    }

    // prestart before chdir, so relative paths on the potential child's
    // command line mean the same thing to the child
    if (!optInitOnly && !optDeleteUpgrade && !optDeleteData &&
	!optDeleteSettings && !optSaveSettings && !optRestoreSettings)
	    if (!prestart(exitCode)) return false;

    sperr << App::appname << QSL(" version " PACKAGE_VERSION) << Qt_endl;
    QDir::setCurrent(dataDir);

    // Make sure we are the only running scheduler using this settings file
    // (after potential detach in prestart())
    settingLockFile = lockLockFile(config->lockFileName());
    if (!settingLockFile) {
	exitCode = SP_EXIT_LOCK_CONFIG;
	return false;
    }

    config->enforce(); // enforce validation on loaded settings

    // Save or restore settings (after getting settings lock)
    if (optSaveSettings || optRestoreSettings) {
	QSettings altSettings(altSettingsFile, QSettings::IniFormat);
	if (optSaveSettings) {
	    altSettings.clear();
	    copySettings(altSettings, *config->settings);
	    qDebug() << "saved settings to" << altSettings.fileName();
	} else {
	    config->settings->clear();
	    copySettings(*config->settings, altSettings);
	    qDebug() << "restored settings from" << altSettings.fileName();
	}
	exitCode = SP_EXIT_OK; // success
	return false;
    }

    // Delete upgrade info (after getting settings lock)
    if (optDeleteUpgrade) {
	qDebug() << "delete-upgrade";
	QDir qdir(dataDir);
	QFile::remove(upFinName());
	config->settings->remove(QSL("upgrade"));
	exitCode = SP_EXIT_OK; // success
	return false; // caller should not do app.exec()
    }

    // Delete data (after getting settings lock)
    if (optDeleteData) {
	QDir qdir(dataDir);
	removeFiles(qdir, proberLogGlob);
	removeFiles(qdir, schedulerLogGlob);
	QFile::remove(dataDir % QSL("/upgrade-log.txt"));
	QString dataDirName = qdir.dirName();
	qdir.cdUp();
	if (qdir.rmdir(dataDirName)) {
	    sperr << "Deleted " << dataDir << Qt_endl;
	    exitCode = SP_EXIT_OK;
	} else {
	    sperr << "Error deleting " << dataDir << ": " <<
		getLastErrmsg() << Qt_endl;
	}
	return false; // caller should not do app.exec()
    }

    // Delete settings (after getting settings lock)
    if (optDeleteSettings) {
	QString filename = config->fileName(); // copy before it disappears
	config->remove();
	sperr << "Deleted settings in " << filename << Qt_endl;
	exitCode = SP_EXIT_OK; // success
	return false; // caller should not do app.exec()
    }

    // Write command line options to settings (after getting settings lock)
    if (optSharePublic >= 0) {
	qDebug() << "config->sharePublic" << (optSharePublic == 1);
	config->sharePublic(optSharePublic == 1);
    }

    if (optShareRemedy >= 0) {
	qDebug() << "config->shareRemedy" << (optShareRemedy == 1);
	config->shareRemedy(optShareRemedy == 1);
    }

    if (optStartPaused >= 0) {
	qDebug() << "config->paused" << (optStartPaused == 1);
	config->paused(optStartPaused == 1);
    }

    if (optInitOnly) {
	config->dataDir(dataDir);
	config->sync();
	exitCode = SP_EXIT_OK; // success
	return false; // caller should not do app.exec()
    }

    // Make sure we are the only running scheduler using this dataDir (in case
    // two different settings files have the same dataDir value, or there's an
    // older version of scheduler that locked only dataDir, not settings)
    dataLockFile = lockLockFile(dataDir % QSL("/spoofer.lock"));
    if (!dataLockFile) {
	exitCode = SP_EXIT_LOCK_DATA;
	return false;
    }

    if (!initSignals()) return false;

    {
	// load history
	config->settings->beginGroup(QSL("history"));
	const QStringList groups = config->settings->childGroups();
	for (const QString &group : groups) {
	    QString subnetstr = group;
	    subnetstr.replace(QSL(";"),QSL("/")); // QSettings doesn't allow "/" in keys
	    if (!subnetstr.contains(QSL("/"))) {
		qDebug() << "removing non-subnet group:" << group;
		config->settings->remove(group);
		continue;
	    }
	    SubnetAddr subnet(QHostAddress::parseSubnet(subnetstr));
	    if (subnet.pfxlen() <= 0) {
		qDebug() << "skipping invalid subnet group:" << group;
		continue;
	    }
	    config->settings->beginGroup(group);
	    RunRecord &rr = pastRuns[subnet];
	    rr.t = static_cast<time_t>(config->settings->value(QSL("time")).toLongLong());
	    rr.errors = config->settings->value(QSL("errors")).toInt();
	    config->settings->endGroup();
	}
	config->settings->endGroup();
    }

    if (!(privServer = listen(true)))
	return false;
#ifndef EVERYONE_IS_PRIVILEGED
    if (!(unprivServer = listen(false)))
	return false;
#endif

    config->dataDir(dataDir);
    config->schedulerSocketName(privServer->fullServerName());
    config->sync();

#ifdef ENABLE_QNetworkConfigurationManager
    // QNetworkConfigurationManager on some platforms does very frequent
    // polling which can be very cpu-intensive and disruptive, maybe even
    // causing some drivers to drop connections.
    ncm = new QNetworkConfigurationManager(this);

    connect(ncm, &QNetworkConfigurationManager::configurationChanged,
	this, &App::handleNetChange);
    connect(ncm, &QNetworkConfigurationManager::configurationAdded,
	this, &App::handleNetChange);
    connect(ncm, &QNetworkConfigurationManager::configurationRemoved,
	this, &App::handleNetChange);
#endif

#if 0 // DEBUGGING: simulate frequent network change signal from OS or Qt
    QTimer *thrashTimer = new QTimer();
    connect(thrashTimer, &QTimer::timeout, this, &App::handleNetChange);
    thrashTimer->start(1000*10);
#endif

    proberTimer.setSingleShot(true);
    connect(&proberTimer, &QTimer::timeout, this, SIGCAST(App, startProber, ()));

    netPollTimer.setSingleShot(true);
    connect(&netPollTimer, &QTimer::timeout, this, &App::scheduleNextProber);

    if (!(paused = config->paused())) {
	// Wait 1s (in case of a burst of changes), then scheduleNextProber()
	netPollTimer.start(1000*1);
    }

    int nlogs = config->keepLogs();
    if (nlogs > 0)
	removeFiles(dir, schedulerLogGlob, nlogs);

    // load upgrade info
    config->settings->beginGroup(QSL("upgrade"));
    if (config->settings->contains(QSL("vnum"))) {
	bool ok;
	int vnum = config->settings->value(QSL("vnum")).toInt(&ok);
	QString upgrade_key = config->settings->value(QSL("upgrade_key")).toString();
	if (!ok || vnum <= NVERSION ||
#ifdef UPGRADE_KEY
	    upgrade_key != QSL(UPGRADE_KEY)
#else
	    upgrade_key != QSL("")
#endif
	) {
	    qDebug() << "ignoring obsolete upgrade" << vnum << "with key" << upgrade_key;
	    config->settings->remove(QSL(""));
	} else {
	    // If an upgrade was recorded in settings, it must be because in a
	    // previous scheduler process either autoupgrade was disabled, the
	    // upgrade was cancelled, or the upgrade failed.  In any case, we
	    // don't want to AUTOupgrade now, but we do want to let users know
	    // it's available.  However, we don't set upgradeIfGreater, so if
	    // a prober gets an upgrade notice (even for the same version), we
	    // may AUTOupgrade then.
	    upgradeInfo = new sc_msg_upgrade_available(false,
		config->settings->value(QSL("mandatory")).toBool(), vnum,
		config->settings->value(QSL("vstr")).toString(),
		config->settings->value(QSL("file")).toString());
	    qDebug() << "upgrade" <<
		(upgradeInfo->mandatory ? "required:" : "available:") <<
		upgradeInfo->vnum;
	}
    }
    config->settings->endGroup();

    return true; // caller should continue with app.exec()
}

void App::cleanup()
{
    if (prober) {
	killProber();
	deleteProber();
    }
    if (privServer) {
	if (config)
	    config->schedulerSocketName.remove();
	delete privServer;
	privServer = nullptr;
    }
    if (unprivServer) {
	delete unprivServer;
	unprivServer = nullptr;
    }
    if (dataLockFile) {
	delete dataLockFile;
	dataLockFile = nullptr;
    }
    if (settingLockFile) {
	delete settingLockFile;
	settingLockFile = nullptr;
    }
#ifdef AUTOUPGRADE_ENABLED
    upgradePromptTimer.stop();
    upgradeState = US_NONE;
#endif
    delete upgradeInfo; upgradeInfo = nullptr;
}

App::~App()
{
    cleanup();
}

void App::sendUpgradeProgress(QLocalSocket *ui, const QString &text)
{
    BlockWriter(ui) << (qint32)SC_UPGRADE_PROGRESS <<
	sc_msg_text(QSL("Upgrade to %1 in progress: %2").arg(upgradeInfo->vstr).arg(text));
}

void App::sendUpgradeError(QLocalSocket *ui, const QString &text)
{
    BlockWriter(ui) << (qint32)SC_UPGRADE_ERROR <<
	sc_msg_text(QSL("Upgrade to %1 failed: %2").arg(upgradeInfo->vstr).arg(text));
}

void App::uiAccept()
{
    QLocalServer *server = static_cast<QLocalServer*>(QObject::sender());
    bool privileged = (server == privServer);
    const char *label = privileged ? "privileged" : "unprivileged";
    qDebug().noquote() << "new" << label << "connection detected on" <<
	server->fullServerName();
    QLocalSocket *ui = server->nextPendingConnection();
    if (!ui) {
	qDebug() << "SCHEDULER ERROR: connection:" << server->errorString();
	return;
    }
    qDebug() << "accepted" << label << "connection";
    connect(ui, &QLocalSocket::readyRead, this, &App::uiRead);
    connect(ui, &QLocalSocket::disconnected, this, &App::uiDelete);
    uiSet.insert(ui);

    BlockWriter(ui) << (qint32)SC_HELLO << sc_msg_text(QSL(PACKAGE_VERSION));

    if (upgradeState == US_DOWNLOADING) {
	sendUpgradeProgress(ui, QSL("Downloading..."));
	return;
    } else if (upgradeState == US_VERIFYING) {
	sendUpgradeProgress(ui, QSL("Verifying..."));
	return;
    } else if (upgradeState == US_INSTALLING) {
	BlockWriter(ui) << (qint32)SC_UPGRADE_INSTALLING;
	return;
    }

    upFinRemover.set(UpFinRemover::uiConnected);

    if (opAllowed(ui, config->unprivPref) && upgradeInfo) {
	if (upgradeInfo->autoTime >= 0) {
	    // reset upgrade timer and notify all UIs
	    promptForUpgrade();
	} else {
	    // notify the new UI
	    BlockWriter(ui) << (qint32)SC_UPGRADE_AVAILABLE << *upgradeInfo;
	}
    }

    if (opAllowed(ui, config->unprivView)) {
	if (paused)
	    BlockWriter(ui) << (qint32)SC_PAUSED;

	if (prober) {
	    // notify new UI of prober in progress
	    BlockWriter(ui) << (qint32)SC_PROBER_STARTED <<
		sc_msg_text(proberOutputFileName);
	} else if (nextProberStart.when) {
	    // notify new UI of next scheduled prober
	    BlockWriter(ui) << (qint32)SC_SCHEDULED << nextProberStart;
	}

	if (opAllowed(ui, config->unprivPref) && !config->hasRequiredSettings())
	    BlockWriter(ui) << (qint32)SC_NEED_CONFIG;
    }
}

void App::uiDelete()
{
    QLocalSocket *ui = static_cast<QLocalSocket*>(sender());
    qDebug() << "UI disconnected";
    uiSet.remove(ui);
    ui->deleteLater();
}

bool App::opAllowed(QLocalSocket *ui, const Config::MemberBase &cfgItem)
{
    if (ui->parent() == privServer) return true;
    if (cfgItem.variant().toBool()) return true;
    return false;
}

bool App::opAllowedVerbose(QLocalSocket *ui, const Config::MemberBase &cfgItem)
{
    if (opAllowed(ui, cfgItem)) return true;
    qDebug() << "Permission denied.";
    BlockWriter(ui) << (qint32)SC_ERROR <<
	sc_msg_text(QSL("Permission denied."));
    return false;
}

void App::uiRead()
{
    static QRegularExpression set_re(QSL("set\\s+(\\w+)\\s+(.*)"));
    QRegularExpressionMatch match;

    qDebug() << "uiRead()";

    QLocalSocket *ui = static_cast<QLocalSocket*>(sender());

    char data[1024];
    qint64 n;
    while ((n = ui->readLine(data, sizeof(data) - 1)) > 0) {
	while (n > 0 && data[n-1] == '\n') data[--n] = '\0';
	sperr << "UI read: " << data << Qt_endl;
	if (data[0] == '#') {
	    // ignore comment
	} else if (strcmp(data, "shutdown") == 0) {
	    sperr << "shutdown by command" << Qt_endl;
	    this->shutdown();
	} else if (strcmp(data, "run") == 0) {
	    if (!opAllowedVerbose(ui, config->unprivTest)) continue;
	    if (prober) {
		BlockWriter(ui) << (qint32)SC_ERROR <<
		    sc_msg_text(QSL("There is already a prober running."));
	    } else if (upgradeState > US_NONE) {
		BlockWriter(ui) << (qint32)SC_ERROR << 
		    sc_msg_text(QSL("Upgrade pending."));
	    } else if (!config->hasRequiredSettings()) {
		BlockWriter(ui) << (qint32)SC_NEED_CONFIG;
	    } else {
		startProber(true);
	    }
	} else if (strcmp(data, "abort") == 0) {
	    if (!opAllowedVerbose(ui, config->unprivTest)) continue;
	    if (!prober) {
		BlockWriter(ui) << (qint32)SC_ERROR <<
		    sc_msg_text(QSL("There is no prober running."));
	    } else {
		killProber();
		// BlockWriter(ui) << (qint32)SC_TEXT <<
		//     sc_msg_text(QSL("Killing prober."));
	    }
	} else if (strcmp(data, "pause") == 0) {
	    if (!opAllowedVerbose(ui, config->unprivPref)) continue;
	    if (paused) {
		const sc_msg_text msg(QSL("The scheduler is already paused."));
		BlockWriter(ui) << (qint32)SC_ERROR << msg;
	    } else {
		pause();
		BlockWriter(ui) << (qint32)SC_DONE_CMD;
	    }
	} else if (strcmp(data, "resume") == 0) {
	    if (!opAllowedVerbose(ui, config->unprivPref)) continue;
	    if (!paused) {
		const sc_msg_text msg(QSL("The scheduler is not paused."));
		BlockWriter(ui) << (qint32)SC_ERROR << msg;
	    } else {
		resume();
		BlockWriter(ui) << (qint32)SC_DONE_CMD;
	    }
	} else if ((match = set_re.match(QString::fromLocal8Bit(data))).hasMatch()) {
	    if (!opAllowedVerbose(ui, config->unprivPref)) continue;
	    QString name = match.captured(1);
	    QString value = match.captured(2);
	    SpooferBase::Config::MemberBase *cfgItem = config->find(name);
	    if (cfgItem) {
		sc_msg_text msg;
		bool wasConfiged = config->hasRequiredSettings();
		if (!cfgItem->setFromString(value, msg.text)) {
		    BlockWriter(ui) << (qint32)SC_ERROR << msg;
		} else {
		    bool isConfiged = config->hasRequiredSettings();
		    config->sync();
		    BlockWriter(ui) << (qint32)SC_DONE_CMD;
		    for (QLocalSocket *connectedUi : uiSet) {
			BlockWriter(connectedUi) << (qint32)SC_CONFIG_CHANGED;
			if (!wasConfiged && isConfiged)
			    BlockWriter(connectedUi) << (qint32)SC_CONFIGED;
		    }
		    // scheduleNextProber() in case any scheduling parameters
		    // changed (but wait 1s in case of a burst of changes)
		    scheduledSubnets.clear(); // force rescheduling
		    netPollTimer.start(1000*1);
		    dataDir = config->dataDir(); // in case it changed
#ifdef AUTOUPGRADE_ENABLED
		    if (!config->autoUpgrade()) cancelUpgrade();
#endif
		}
	    } else {
		BlockWriter(ui) << (qint32)SC_ERROR <<
		    sc_msg_text(QSL("No such setting \"%1\".").arg(name));
	    }
	} else if (strcmp(data, "sync") == 0) {
	    if (!opAllowedVerbose(ui, config->unprivPref)) continue;
	    config->sync();
	    BlockWriter(ui) << (qint32)SC_DONE_CMD;
	} else if (strcmp(data, "dumpnetcfg") == 0) {
	    dumpNetCfg(ui);
	    BlockWriter(ui) << (qint32)SC_DONE_CMD;
#ifdef AUTOUPGRADE_ENABLED
	} else if (strcmp(data, "upgrade") == 0) {
	    if (!opAllowedVerbose(ui, config->unprivPref)) continue;
	    if (upgradeInfo) {
		startUpgrade();
	    } else {
		qDebug() << "No upgrade available.";
		const sc_msg_text msg(QSL("No upgrade available."));
		BlockWriter(ui) << (qint32)SC_ERROR << msg;
	    }
	} else if (strcmp(data, "cancel") == 0) {
	    cancelUpgrade();
#endif
	} else {
	    qDebug() << "Unknown command:" << data;
	    BlockWriter(ui) << (qint32)SC_ERROR << 
		sc_msg_text(QSL("Unknown command."));
	}
    }
    if (n < 0) {
	qDebug() << "UI read error:" << ui->errorString();
    }
}

void App::startProber(bool manual)
{
    proberTimer.stop();
    nextProberStart.when = 0;
    if (prober) {
	qDebug() << "startProber: there is already a prober running.";
	return;
    } else if (upgradeState > US_PROMPTED) {
	qDebug() << "startProber: upgrade in progress.";
	return;
    }
    QDir dir(dataDir);
    proberOutputFileName = dir.filePath(ftime_utc(proberLogFtime));

    prober = new QProcess(this);
    prober->setProperty("manual", QVariant(manual));
    prober->setProcessChannelMode(QProcess::MergedChannels); // ... 2>&1
    prober->setStandardInputFile(QProcess::nullDevice()); // ... </dev/null
    prober->setStandardOutputFile(proberOutputFileName,
	QIODevice::Text | QIODevice::Truncate | QIODevice::WriteOnly); // ... >file
    QStringList args;
#if DEBUG
    if (config->useDevServer()) args << QSL("-T");
    if (config->spooferProtocolVersion())
        args << QSL("-V") << QString::number(config->spooferProtocolVersion());
    if (config->pretendMode()) args << QSL("-P");
    if (config->standaloneMode()) args << QSL("-S");
#endif
    args << (config->sharePublic() ? QSL("-s1") : QSL("-s0"));
    args << (config->shareRemedy() ? QSL("-r1") : QSL("-r0"));
    if (config->enableIPv4()) args << QSL("-4");
    if (config->enableIPv6()) args << QSL("-6");
    if (!config->enableTLS()) args << QSL("--no-tls");
    if (config->singleThreaded()) args << QSL("-1");
#ifdef AUTOUPGRADE_ENABLED
    upgradeState = US_NONE;
    if (config->autoUpgrade() && config->enableTLS())
	// Without TLS, we can't completely trust the upgrade message
	args << QSL("-U") << QString().setNum(upgradeIfGreater);
#endif
    sperr << "prober args: " << args.join(QSL(" ")) << Qt_endl;
    sperr << "prober output: " << QDir::toNativeSeparators(proberOutputFileName)
	<< Qt_endl;

    connect(prober, &QProcess::started, this, &App::proberStarted);
    connect(prober, QPROCESS_ERROR_OCCURRED, this, &App::proberError);
    connect(prober, SIGCAST(QProcess, finished, (int,QProcess::ExitStatus)),
	this, &App::proberFinished);

    QString proberPath = appDir % QSL("/spoofer-prober");
    prober->start(proberPath, args);
    // Unfortunately, if this scheduler process exits, ~QProcess() will kill
    // the prober process.  QProcess::startDetached() would avoid that, but
    // then we couldn't set up its input/output and signals/slots.

    // If prober output file goes unmodified for 5 minutes, assume the prober
    // is hung, and kill it.
    proberWatcher = new QFileSystemWatcher(this);
    proberWatcher->addPath(proberOutputFileName);
    hangTimer = new QTimer(this);
    hangTimer->setSingleShot(true);
    hangTimer->start(5 * 60 * 1000); // 5 minutes
    connect(hangTimer, &QTimer::timeout, this, &App::killProber);
    connect(proberWatcher, &QFileSystemWatcher::fileChanged,
	hangTimer, SIGCAST(QTimer, start, ()));
}

void App::deleteProber()
{
    delete hangTimer; hangTimer = nullptr;
    delete proberWatcher; proberWatcher = nullptr;
    prober->deleteLater(); prober = nullptr;
}

void App::killProber()
{
    if (!prober) return;
    qWarning() << "terminating prober.";
    prober->terminate();
    if (!prober->waitForFinished(1000) &&
	prober->state() != QProcess::NotRunning)
    {
	qWarning() << "prober did not terminate; killing by force.";
	prober->kill();
    }
}

void App::proberStarted()
{
    qint64 pid =
#if QT_VERSION >= 0x050300 // 5.3.0 or later
	prober->processId();
#elif defined(Q_OS_WIN32)
	prober->pid()->dwProcessId;
#else
	prober->pid();
#endif
    sperr << "prober started, pid " << pid << Qt_endl;
    const sc_msg_text msg(proberOutputFileName);
    for (QLocalSocket *ui : uiSet) {
	if (opAllowed(ui, config->unprivView))
	    BlockWriter(ui) << (qint32)SC_PROBER_STARTED << msg;
    }
    upFinRemover.set(UpFinRemover::proberStarted);
}

void App::proberError(QProcess::ProcessError e)
{
    Q_UNUSED(e); // processErrorMessage() gets error from p
    // If this error is not the first, this->prober was aleady nulled.
    QProcess *p = dynamic_cast<QProcess*>(sender());

    QString msg = p->program() % QSL(": ") % processErrorMessage(*p);
    qCritical() << "prober error:" << qPrintable(msg);
    if (p->state() == QProcess::NotRunning) {
	sc_msg_text scmsg(msg);
	for (QLocalSocket *ui : uiSet) {
	    if (opAllowed(ui, config->unprivView))
		BlockWriter(ui) << (qint32)SC_PROBER_ERROR << scmsg;
	}
	if (prober) {
	    qDebug() << "delete prober";
	    recordRun(false); // before deleting prober
	    deleteProber();
	    scheduleNextProber();
	}
    }
}

void App::proberFinished(int exitCode, QProcess::ExitStatus exitStatus)
{
    char buf[80];
    bool success = false;
    qDebug() << "prober finished" << qPrintable(ftime_utc());

    sc_msg_text msg;
    if (exitStatus == QProcess::NormalExit) {
	sperr << "prober exited normally, exit code " << exitCode << Qt_endl;
	snprintf(buf, sizeof(buf), "normally (exit code %d).", exitCode);
	msg.text = QString::fromLocal8Bit(buf);
	success = (exitCode == SP_EXIT_OK);
    } else {
	sperr << "prober exited abnormally" << Qt_endl;
	msg.text = QSL("abnormally.");
    }

    for (QLocalSocket *ui : uiSet) {
	if (opAllowed(ui, config->unprivView))
	    BlockWriter(ui) << (qint32)SC_PROBER_FINISHED <<
		exitCode << (int)exitStatus;
    }

    recordRun(success); // before deleting prober
    deleteProber();

    // Prober must be done before we can start upgrade.
    // UIs expect SC_UPGRADE_AVAILABLE after SC_PROBER_FINISHED
    parseProberTextForUpgrade();

    QDir dir(dataDir);
    int nlogs = config->keepLogs();
    if (nlogs > 0) {
	removeFiles(dir, proberLogGlob, nlogs);
    }

    if (upgradeState == US_NONE)
	scheduleNextProber();
}

void App::recordRun(bool success)
{
    const QList<SubnetAddr> addrs = getAddresses();
    time_t now;
    time(&now);
    bool manual = prober && prober->property("manual").toBool();
    if (!success && manual) {
	// don't let failed manual prober affect scheduling
	qDebug().noquote() << "not recording failed manual prober";
	return;
    } else {
	qDebug().noquote() << QSL("recording %1 %2 prober")
	    .arg(success ? QSL("successful") : QSL("failed"))
	    .arg(manual ? QSL("manual") : QSL("scheduled"));
    }
    config->settings->beginGroup(QSL("history"));
    for (const SubnetAddr &addr : addrs) {
	SubnetAddr subnet = addr.prefix();
	RunRecord &rr = pastRuns[subnet];
	if (rr.t == now) continue; // don't duplicate
	rr.t = now;
	if (success)
	    rr.errors = 0;
	else if (upgradeState > US_NONE)
	    // make prober run shortly after upgrade finishes or is canceled
	    rr.errors = 1;
	else
	    rr.errors++;
	QString subnetKey = subnet.toString().replace(QSL("/"),QSL(";"));
	config->settings->beginGroup(subnetKey);
	config->settings->setValue(QSL("time"), (qlonglong)rr.t);
	config->settings->setValue(QSL("errors"), rr.errors);
	config->settings->endGroup();
    }
    config->settings->endGroup();
    config->sync();
}

void App::recordUpgrade()
{
    config->settings->beginGroup(QSL("upgrade"));
#ifdef UPGRADE_KEY
    config->settings->setValue(QSL("upgrade_key"), QSL(UPGRADE_KEY));
#endif
    config->settings->setValue(QSL("vstr"), upgradeInfo->vstr);
    config->settings->setValue(QSL("vnum"), (qlonglong)upgradeInfo->vnum);
    config->settings->setValue(QSL("file"), upgradeInfo->file);
    config->settings->setValue(QSL("mandatory"), upgradeInfo->mandatory);
    config->settings->endGroup();
    config->sync();
}

void App::pause()
{
    if (!config->paused()) {
	config->paused(true);
	config->sync();
    }
    if (paused) return;
    sperr << "pausing" << Qt_endl;
    paused = true;
    proberTimer.stop();
    netPollTimer.stop();
    nextProberStart.when = 0;
    scheduledSubnets.clear();
    for (QLocalSocket *ui : uiSet) {
	if (opAllowed(ui, config->unprivView))
	    BlockWriter(ui) << (qint32)SC_PAUSED;
    }
}

void App::resume()
{
    if (config->paused()) {
	config->paused(false);
	config->sync();
    }
    if (!paused) return;
    sperr << "resuming" << Qt_endl;
    paused = false;
    for (QLocalSocket *ui : uiSet) {
	if (opAllowed(ui, config->unprivView))
	    BlockWriter(ui) << (qint32)SC_RESUMED;
    }
    scheduleNextProber();
}

void App::parseProberTextForUpgrade()
{
    static QRegularExpression re(QSL(
	">>   enableTLS: (?<tls>yes)|"
	"Upgrade ((?<mandatory>required)|available):\\s+(?<vstr>\\S+) "
	"\\((?<vnum>\\d+)\\) at (?<file>\\S+)\\s*$"
    ));

    QFile file(proberOutputFileName);
    if (!file.open(QIODevice::ReadOnly | QIODevice::Text)) {
	qWarning().noquote() << "Failed to open" << proberOutputFileName <<
	    ":" << file.errorString();
	return;
    }
    QTextStream in(&file);
    QString text;
    bool found = false;
    bool haveTLS = false;
    while (!in.atEnd()) {
#if QT_VERSION >= 0x050500 // Qt 5.5 or later
	in.readLineInto(&text);
#else
	text = in.readLine();
#endif
	QRegularExpressionMatch match = re.match(text);
	if (!match.captured(QSL("tls")).isNull())
	    haveTLS = true; // we can trust an upgrade message
	if (!match.captured(QSL("vstr")).isNull()) {
	    if (upgradeInfo) delete upgradeInfo;
	    found = true;
	    int vnum = match.captured(QSL("vnum")).toInt();
	    bool wantAuto = false;
#ifdef AUTOUPGRADE_ENABLED
	    wantAuto = config->autoUpgrade() && (vnum > upgradeIfGreater);
#endif
	    upgradeInfo = new sc_msg_upgrade_available(
		wantAuto && haveTLS,
		!match.captured(QSL("mandatory")).isNull(),
		vnum,
		match.captured(QSL("vstr")),
		match.captured(QSL("file")));
	    break;
	}
    }

    if (found) {
	sperr << "Upgrade " <<
	    (upgradeInfo->mandatory ? "required" : "available") << ": " <<
	    upgradeInfo->vstr << " (v=" << upgradeInfo->vnum << ", t=" <<
	    upgradeInfo->autoTime << ") at " << upgradeInfo->file << Qt_endl;
	recordUpgrade();
	if (nextProberStart.when > 0) { // disable scheduled prober run
	    proberTimer.stop();
	    netPollTimer.stop();
	    nextProberStart.when = 0;
	    for (QLocalSocket *ui : uiSet) {
		if (opAllowed(ui, config->unprivView))
		    BlockWriter(ui) << (qint32)SC_SCHEDULED << nextProberStart;
	    }
	}
	promptForUpgrade();
    } else {
	qDebug().noquote() << "No upgrade info found in" << proberOutputFileName;
    }
}

void App::promptForUpgrade()
{
    int prompts = 0;
    for (QLocalSocket *ui : uiSet) {
	if (opAllowed(ui, config->unprivPref)) {
	    BlockWriter(ui) << (qint32)SC_UPGRADE_AVAILABLE << *upgradeInfo;
	    prompts++;
	}
    }
#ifdef AUTOUPGRADE_ENABLED
    if (upgradeInfo->autoTime >= 0) {
	if (prompts > 0) {
	    upgradeState = US_PROMPTED;
	    upgradePromptTimer.start(upgradeInfo->autoTime * 1000);
	} else {
	    startUpgrade();
	}
    }
#endif
}

#ifdef AUTOUPGRADE_ENABLED
void App::cancelUpgrade()
{
    if (upgradeState == US_NONE)
	return;
    upgradeInfo->autoTime = -1;
    upgradePromptTimer.stop();
    if (downloader) {
	qDebug() << "cancelUpgrade(): aborting downloader";
	downloader->abort(QSL("Upgrade cancelled during download."));
	// abort will trigger downloadFailed()
    } else {
	// Change upgradeIfGreater here so that the next prober (possibly
	// after just proberRetryInterval) doesn't skip its probing because of
	// the same upgrade that we're canceling here.
	if (upgradeIfGreater < upgradeInfo->vnum)
	    upgradeIfGreater = upgradeInfo->vnum;
	upgradeState = US_NONE;
	for (QLocalSocket *ui : uiSet) {
	    sendUpgradeError(ui, QSL("Upgrade cancelled."));
	}
	promptForUpgrade(); // upgrade is still available, just not automatic
	scheduleNextProber();
    }
}

void App::startUpgrade()
{
    if (!upgradeInfo) {
	for (QLocalSocket *ui : uiSet) {
	    sendUpgradeError(ui, QSL("Internal error: startUpgrade() without upgradeInfo"));
	}
	qWarning() << "Internal error: startUpgrade() without upgradeInfo";
	return;
    }

    // Never automatically retry this failed upgrade version (in this
    // scheduler's lifetime), so we don't get stuck in a loop of failed
    // upgrades and failed probers.
    if (upgradeIfGreater < upgradeInfo->vnum)
        upgradeIfGreater = upgradeInfo->vnum;

    sperr << "Upgrading to " << upgradeInfo->vstr << Qt_endl;
    upgradeInfo->autoTime = -1;
    upgradePromptTimer.stop();
    QFile::remove(upFinName());
    upFinRemover.reset();
    QFile::remove(dataDir % QSL("/upgrade-log.txt"));
    if (prober) {
	for (QLocalSocket *ui : uiSet) {
	    sendUpgradeError(ui, QSL("Can't upgrade while prober is running."));
	}
	qWarning() << "Can't upgrade while prober is running.";
	return;
    }

#ifdef UPGRADE_WITHOUT_DOWNLOAD
    // execute an installer that does not need to be downloaded first
    startInstaller(upgradeInfo->file);
#else
    // download installer, then execute it with startInstaller()
    sperr << "Downloading " << upgradeInfo->file << Qt_endl;
    for (QLocalSocket *ui : uiSet) {
	sendUpgradeProgress(ui, QSL("Downloading..."));
	ui->waitForBytesWritten(); // write now, before returning to event loop
	qDebug() << "sent SC_UPGRADE_PROGRESS \"Downloading\" to ui";
    }
    upgradeState = US_DOWNLOADING;
    downloader = new Downloader(upgradeInfo->file, dataDir);
    downloader->addTaint = config->installerAddTaint();
    downloader->verifySig = config->installerVerifySig();
    connect(downloader, &Downloader::finished, this, &App::downloadFinished);
    connect(downloader, &Downloader::error, this, &App::downloadFailed);
    connect(downloader, &Downloader::verifying, this, &App::enterUpgradeVerifyState);
    downloader->start();
#endif // UPGRADE_WITHOUT_DOWNLOAD
}

void App::enterUpgradeVerifyState()
{
    upgradeState = US_VERIFYING;
    for (QLocalSocket *ui : uiSet) {
	sendUpgradeProgress(ui, QSL("Verifying..."));
	ui->waitForBytesWritten(); // write now, before returning to event loop
	qDebug() << "sent SC_UPGRADE_PROGRESS \"Verifying\" to ui";
    }
}

void App::downloadFailed()
{
    abortInstallation(downloader->errorString());
    downloader->deleteLater();
    downloader = nullptr;
}

void App::installerCheck()
{
    if (!installerIsRunning()) {
	abortInstallation(QSL("Installer crashed."));
    } else if (++installerTicker > 30) {
	killInstaller();
	abortInstallation(QSL("Installer timed out."));
    }
}

void App::abortInstallation(const QString &err)
{
    qCritical().noquote() << err;
    for (QLocalSocket *ui : uiSet) {
	sendUpgradeError(ui, err);
	qDebug() << "sent SC_UPGRADE_ERROR to ui";
    }
    if (upgradeState >= US_INSTALLING) {
	// UIs have been stopped, so we must write something to trigger them
	// to restart in case the upgrade process hasn't written anything yet.
	QFile upFinFile(upFinName());
	if (!upFinFile.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Append)) {
	    qCritical().noquote() << "failed to open" << upFinName() << ":" <<
		upFinFile.errorString();
	} else {
	    QTextStream out(&upFinFile);
	    out << "Upgrade failed: " << err << Qt_endl;
	    upFinFile.close();
	}
    }
    upgradeState = US_NONE;
    if (installerTimer) {
	installerTimer->stop();
	installerTimer->deleteLater();
	installerTimer = nullptr;
    }
    promptForUpgrade(); // upgrade is still available, just not automatic
    scheduleNextProber();
}

void App::downloadFinished()
{
    QString installerName = downloader->fileName();
    sperr << "Downloaded " << downloader->byteCount() << " bytes to " <<
	installerName << Qt_endl;
    downloader->deleteLater();
    downloader = nullptr;
    startInstaller(installerName);
}

void App::startInstaller(const QString &installerName)
{
    installerTimer = new QTimer(this);
    connect(installerTimer, &QTimer::timeout, this, &App::installerCheck);
    installerTicker = 0;
    installerTimer->start(1000);

    upgradeState = US_INSTALLING;
    for (QLocalSocket *ui : uiSet) {
	BlockWriter(ui) << (qint32)SC_UPGRADE_INSTALLING;
	ui->waitForBytesWritten(); // write now, before returning to event loop
	qDebug() << "sent SC_UPGRADE_INSTALLING to ui";
    }

    executeInstaller(installerName);

    // The installer we just started will kill this scheduler & start a new one.
}
#endif // AUTOUPGRADE_ENABLED
