/* 
 * Copyright 2015-2022, 2026 The Regents of the University of California
 * All rights reserved.
 * 
 * This file is part of Spoofer.
 * 
 * Spoofer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Spoofer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Spoofer.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <unistd.h>
#include <errno.h>
#include <launch.h>
#include <fcntl.h> // open flags in posix_spawn()
#include <spawn.h> // posix_spawn()
#if __MAC_OS_X_VERSION_MIN_REQUIRED >= 101200 // 10.12;
 #include <sysdir.h> // sysdir_start_search_path_enumeration()
#else
 #include <NSSystemDirectories.h> // NSStartSearchPathEnumeration()
#endif
#include "spoof_qt.h"
#include <QFileSystemWatcher>
#include "../../config.h"
#include "app.h"
#include "appmac.h"
#include "common.h"

bool AppMac::prestart(int &exitCode)
{
    // NB: ppid may change after daemonizing, so we must check it now
    isLaunchdService = (getppid() == 1);

    if (isLaunchdService) {
	if (optDetach) {
	    qDebug() << "Scheduler: ignoring detatch option";
	    optDetach = false;
	}
	pApplabel = QSL(" service");

	// Try to detect if our own executable is deleted (i.e. user is trying
	// to uninstall by dragging the Spoofer.app bundle to the trash).
	QString bundleDir(appDir);
	bundleDir.remove(QSL("/Contents/MacOS")); // -> "/Applications/Spoofer.app"
	QString bundleParent(bundleDir % QSL("/.."));
	exeWatcher = new QFileSystemWatcher();
	QStringList list = (QStringList() <<
	    appFile << bundleDir << bundleParent);
	qDebug() << "watching" << list;
	QStringList failed = exeWatcher->addPaths(list);
	if (!failed.isEmpty()) {
	    qDebug() << "failed to watch" << failed;
	}
	if (failed.size() == list.size()) {
	    delete exeWatcher;
	} else {
	    connect(exeWatcher, &QFileSystemWatcher::fileChanged,
		this, &AppMac::executableChanged);
	    connect(exeWatcher, &QFileSystemWatcher::directoryChanged,
		this, &AppMac::executableChanged);
	}

    } else if (optDetach) {
	// A macOS process using threads (as we do inside Qt) will crash after
	// fork().  (See http://www.evanjones.ca/fork-is-dangerous.html).
	// Instead, we use posix_spawn() to start a detached child process
	// with the same command line as this one (minus the daemon option).
	// (QProcess::startDetached() might also work, but the version that
	// allows i/o redirection wasn't added until Qt 5.10.)
	pid_t childPid = 0;
	posix_spawn_file_actions_t f_actions;
	posix_spawnattr_t attr;
	extern char **environ;

	posix_spawnattr_init(&attr);
	posix_spawnattr_setflags(&attr,
	    POSIX_SPAWN_CLOEXEC_DEFAULT | // close other files (Apple extension)
	    POSIX_SPAWN_SETPGROUP);       // set process group

	posix_spawn_file_actions_init(&f_actions);

	QString logname;
	int logfd = -1;
	if (errdev.type() == typeid(Syslog)) {
	    // parent was using syslog; child continues to use syslog
	    logname = QSL("(syslog)");
	} else {
	    // parent was using logfile or stderr; child uses logfile
	    logname = AppLog::makeName();
	    // Note: ..._file_actions_addopen() wouldn't let us report errors
	    logfd = open(logname.toLocal8Bit().constData(), O_CREAT|O_WRONLY,
		S_IWUSR|S_IRUSR|S_IRGRP|S_IROTH);
	    if (logfd < 0) {
		sperr << "Can't open " << logname << ": " <<
		    lastErrorString() << Qt_endl;
		return false;
	    }
	    posix_spawn_file_actions_adddup2(&f_actions, logfd, STDOUT_FILENO);
	    posix_spawn_file_actions_adddup2(&f_actions, logfd, STDERR_FILENO);
	}

	qDebug() << "spawning:" << appFile;
	QStringList origArgs = QCoreApplication::arguments();
	char **argv = new char*[origArgs.length() + 1];
	argv[0] = strdup(appFile.toLocal8Bit().constData()); // origArgs[0] may not be full path
	int argc = 1;
	for (int i = 1; i < origArgs.length(); i++) {
	    if (origArgs[i] == QSL("-D") || origArgs[i] == QSL("--daemon"))
		continue; // don't daemonize again
	    argv[argc] = strdup(origArgs[i].toLocal8Bit().constData());
	    qDebug() << "   " << argv[argc];
	    argc++;
	}
	argv[argc++] = nullptr;

	if (posix_spawn(&childPid, appFile.toLocal8Bit().constData(),
	    &f_actions, &attr, argv, environ) == 0)
	{
	    qDebug() << "Detached; child pid =" << childPid;
	    sperr << "Log file: " << logname << Qt_endl;
	} else {
	    sperr << "Failed to daemonize: " << lastErrorString() << Qt_endl;
	}
	if (logfd >= 0) close(logfd);
	posix_spawnattr_destroy(&attr);
	posix_spawn_file_actions_destroy(&f_actions);
	if (!childPid) return false; // spawn failed

	pApplabel = QSL(" parent");
	exitCode = verifyDaemon(childPid) ? SP_EXIT_OK : SP_EXIT_DAEMON_FAILED;
	return false; // skip app.exec()
    }

    return AppUnix::prestart(exitCode);
}

#if __MAC_OS_X_VERSION_MIN_REQUIRED < 101200 // 10.12;
 // implement newer sysdir.h API with older NSSystemDirectories.h
 #define sysdir_start_search_path_enumeration(dir, domain) \
     NSStartSearchPathEnumeration(dir, domain)
 #define sysdir_get_next_search_path_enumeration(state, path) \
     NSGetNextSearchPathEnumeration(state, path)
 #define SYSDIR_DIRECTORY_APPLICATION           NSApplicationDirectory
 //#define SYSDIR_DIRECTORY_DEMO_APPLICATION      NSDemoApplicationDirectory
 //#define SYSDIR_DIRECTORY_DEVELOPER_APPLICATION NSDeveloperApplicationDirectory
 #define SYSDIR_DIRECTORY_ADMIN_APPLICATION     NSAdminApplicationDirectory
 #define SYSDIR_DIRECTORY_LIBRARY               NSLibraryDirectory
 //#define SYSDIR_DIRECTORY_DEVELOPER             NSDeveloperDirectory
 #define SYSDIR_DIRECTORY_USER                  NSUserDirectory
 #define SYSDIR_DIRECTORY_DOCUMENTATION         NSDocumentationDirectory
 //#define SYSDIR_DIRECTORY_DOCUMENT              NSDocumentDirectory
 #define SYSDIR_DIRECTORY_CORESERVICE           NSCoreServiceDirectory
 //#define SYSDIR_DIRECTORY_AUTOSAVED_INFORMATION NSAutosavedInformationDirectory
 //#define SYSDIR_DIRECTORY_DESKTOP               NSDesktopDirectory
 #define SYSDIR_DIRECTORY_CACHES                NSCachesDirectory
 #define SYSDIR_DIRECTORY_APPLICATION_SUPPORT   NSApplicationSupportDirectory
 #define SYSDIR_DIRECTORY_DOWNLOADS             NSDownloadsDirectory
 #define SYSDIR_DIRECTORY_INPUT_METHODS         NSInputMethodsDirectory
 //#define SYSDIR_DIRECTORY_MOVIES                NSMoviesDirectory
 //#define SYSDIR_DIRECTORY_MUSIC                 NSMusicDirectory
 //#define SYSDIR_DIRECTORY_PICTURES              NSPicturesDirectory
 //#define SYSDIR_DIRECTORY_PRINTER_DESCRIPTION   NSPrinterDescriptionDirectory
 #define SYSDIR_DIRECTORY_SHARED_PUBLIC         NSSharedPublicDirectory
 #define SYSDIR_DIRECTORY_PREFERENCE_PANES      NSPreferencePanesDirectory
 #define SYSDIR_DIRECTORY_ALL_APPLICATIONS      NSAllApplicationsDirectory
 #define SYSDIR_DIRECTORY_ALL_LIBRARIES         NSAllLibrariesDirectory
 #define SYSDIR_DOMAIN_MASK_LOCAL               NSLocalDomainMask
#endif

void AppMac::dumpPaths() const
{
    char path[PATH_MAX] = "";
    App::dumpPaths();
    spout << "Local Mac paths:" << Qt_endl;

#define dumpMacPath(id)   do { \
    auto state = sysdir_start_search_path_enumeration( \
	SYSDIR_DIRECTORY_##id, SYSDIR_DOMAIN_MASK_LOCAL); \
    spout << #id << Qt_endl; \
    while ((state = sysdir_get_next_search_path_enumeration(state, path))) { \
	QString qpath = QString::fromLocal8Bit(path); \
	spout << "    " << qpath << Qt_endl; \
    } \
} while (0)

    dumpMacPath(APPLICATION);
    // dumpMacPath(DEMO_APPLICATION);
    // dumpMacPath(DEVELOPER_APPLICATION);
    dumpMacPath(ADMIN_APPLICATION);
    dumpMacPath(LIBRARY);
    // dumpMacPath(DEVELOPER);
    dumpMacPath(USER);
    dumpMacPath(DOCUMENTATION);
    //dumpMacPath(DOCUMENT);
    dumpMacPath(CORESERVICE);
    //dumpMacPath(AUTOSAVED_INFORMATION);
    //dumpMacPath(DESKTOP);
    dumpMacPath(CACHES);
    dumpMacPath(APPLICATION_SUPPORT);
    dumpMacPath(DOWNLOADS);
    dumpMacPath(INPUT_METHODS);
    // dumpMacPath(MOVIES);
    // dumpMacPath(MUSIC);
    // dumpMacPath(PICTURES);
    // dumpMacPath(PRINTER_DESCRIPTION);
    dumpMacPath(SHARED_PUBLIC);
    dumpMacPath(PREFERENCE_PANES);
    dumpMacPath(ALL_APPLICATIONS);
    dumpMacPath(ALL_LIBRARIES);
}

QString AppMac::chooseDataDir()
{
    // We want only system-wide dirs, but QStandardPaths::standardLocations()
    // doesn't separate user dirs from system-wide dirs.
    char path[PATH_MAX] = "";
    auto state = sysdir_start_search_path_enumeration(
	SYSDIR_DIRECTORY_APPLICATION_SUPPORT, SYSDIR_DOMAIN_MASK_LOCAL);
    QString result;
    while ((state = sysdir_get_next_search_path_enumeration(state, path))) {
	QString qpath = QString::fromLocal8Bit(path) %
	    QSL("/") % QCoreApplication::organizationName() %
	    QSL("/") % QCoreApplication::applicationName();
	qDebug() << "dataDir option:" << qpath;
	if (result.isEmpty()) result = qpath;
    }
    return result;
}

#define SERVICENAME ORG_DOMAIN_REVERSED ".spoofer-scheduler"
#define LAUNCHERCFG "/Library/LaunchDaemons/" SERVICENAME ".plist"

void AppMac::removeLaunchdService()
{
    QStringList args;
    QProcess proc;
    proc.setProcessChannelMode(QProcess::MergedChannels); // ... 2>&1
    proc.setStandardInputFile(QProcess::nullDevice()); // ... </dev/null
    args << QSL("-c") <<
        QSL("/bin/launchctl bootout system/%1").arg(QSL(SERVICENAME));
    qDebug() << "running: /bin/sh" << args;
    proc.start(QSL("/bin/sh"), args);
    if (!proc.waitForStarted() || !proc.waitForFinished()) {
	sperr << QSL("Could not stop scheduler: %1: %2").arg(
	    proc.program(), processErrorMessage(proc)) << Qt_endl;
	return;
    }
    if (proc.exitStatus() != QProcess::NormalExit) {
	sperr << QSL("Could not stop scheduler: %1 exited abnormally").
	    arg(proc.program()) << Qt_endl;
	return;
    }
    QByteArray output = proc.readAll();
    // Note: even when working correctly, bootout exits with code 36
    // "Operation now in progress" and doesn't wait for service to stop.
    sperr << QSL("Stopping scheduler: %2 (exit code %1)")
        .arg(proc.exitCode()).arg(QString::fromUtf8(output).trimmed()) <<
        Qt_endl;
}

void AppMac::executableChanged(const QString &path)
{
    qDebug() << "change detected in" << path;
    if (QFile(appFile).exists())
	return;
    sperr << "appFile " << appFile << " was deleted" << Qt_endl;

    if (!isLaunchdService)
	return;

    removeLaunchdService();

    // delete our launchd config file
    if (unlink(LAUNCHERCFG) == 0)
	sperr << "Deleted " << LAUNCHERCFG << Qt_endl;
    else
	sperr << "Error deleting " << LAUNCHERCFG << ": " << strerror(errno) <<
	    Qt_endl;

    this->exit(1);
}

void AppMac::shutdown()
{
    // If we're running under launchd, remove ourselves so we're not
    // automatically restarted as soon as we exit.  But we don't delete the
    // config file, so we can be reloaded later (e.g. at next boot).
    if (isLaunchdService)
	removeLaunchdService();
    App::shutdown();
}

